// ClassDescription.js Ver.2.1.0
// MIT License (C) 2022 あわやまたな
// http://opensource.org/licenses/mit-license.php

/*:
* @target MZ
* @plugindesc 職業の説明を追加します。
* @author あわやまたな (Awaya_Matana)
* @url https://awaya3ji.seesaa.net/
* @help 職業のメモ欄に<desc:なんかいい感じの説明>と書くことで
* ステータスのヘルプ欄に職業の説明が表示できます。
*
* プロフィール欄は通常２行までしか書けませんが、
* アクターのメモ欄に<profile:なんかいい感じの説明>と書くことで
* ３行目以降も書けます。
*
* [更新履歴]
* 2022/05/08：Ver.1.0.0　公開
* 2022/05/31：Ver.1.1.0　MVを参考に見た目を改善。
* 2022/06/01：Ver.1.1.1　ライン幅を微調整。
* 2022/08/28：Ver.1.2.0　ウィンドウを分離するモードを追加。
* 2022/09/18：Ver.2.0.0　ツクールフェス風の形式を追加。改行コード対応。
* 2022/12/22：Ver.2.1.0　改行コード機能削除。メモ欄を利用した代替機能を追加。
*
* @param status
* @text ステータス
* @desc ステータスの項目名。形式が拡張の時のみ有効。
* @default ステータス
*
* @param profile
* @text プロフィール
* @desc ヘルプ欄に表示される項目名
* @default プロフィール
*
* @param classDesc
* @text 職業
* @desc ヘルプ欄に表示される項目名
* @default 職業
*
* @param type
* @text 形式
* @desc ウィンドウ配置の変更。
* @default 2
* @type select
* @option 分離
* @value 0
* @option 結合
* @value 1
* @option 拡張
* @value 2
*
* @param shorterTabWindow
* @text 項目選択ウィンドウ高さを低くする。
* @desc 項目選択ウィンドウの高さを低くし、スペースに余裕を持たせます。形式が分離または結合の時のみ。
* @default true
* @type boolean
*
* @param drawItemFrame
* @text 項目枠描画
* @desc 選択肢に枠を描画します。形式が結合の時のみ。
* @default true
* @type boolean
*
* @param horzLine
* @text 水平線
* @desc 項目名とプロフィールを区切る線。形式が結合の時のみ。
* @default true
* @type boolean
*
* @param numLines
* @text 行数
* @desc ステータス、装備、プロフィール欄の行数。形式が拡張の時のみ。
* @default 8
* @type number
*
*/

'use strict';
{
	const pluginName = document.currentScript.src.match(/^.*\/(.*).js$/)[1];
	const parameter = PluginManager.parameters(pluginName);
	const statusText = parameter["status"];
	const profText = parameter["profile"];
	const classText = parameter["classDesc"];
	const type = Number(parameter["type"]);
	const shorterTabWindow = parameter["shorterTabWindow"] === "true";
	const drawItemFrame = parameter["drawItemFrame"] === "true";
	const horzLine = parameter["horzLine"] === "true";
	const numLines = Number(parameter["numLines"]);

	//-----------------------------------------------------------------------------
	// Game_Actor

	const _Game_Actor_setup = Game_Actor.prototype.setup;
	Game_Actor.prototype.setup = function(actorId) {
		_Game_Actor_setup.call(this, actorId);
		const actor = $dataActors[actorId];
		if (actor.meta && actor.meta.profile) {
			if (this._profile) {
				this._profile += "\n";
			}
			this._profile += actor.meta.profile;
		}
	};

	//-----------------------------------------------------------------------------
	// Window_Status

	Window_Status.prototype.drawBlock1 = function() {
		const y = this.block1Y();
		this.drawActorFace(this._actor, 4, y);
		this.drawBasicInfo(ImageManager.faceWidth + 20, y);
		this.drawExpInfo(528, y);
	};

	Window_Status.prototype.block1Y = function() {
		return Math.floor((this.innerHeight - ImageManager.faceHeight)/2);
	};

	Window_Status.prototype.drawBlock2 = function() {};

	Window_Status.prototype.drawBasicInfo = function(x, y) {
		const actor = this._actor;
		const lineHeight = this.lineHeight();
		const x2 = x + 180;
		this.drawActorName(actor, x, y);
		this.drawActorNickname(actor, x, y + lineHeight);
		this.drawActorLevel(actor, x, y + lineHeight * 2);
		this.drawActorIcons(actor, x, y + lineHeight * 3);
		this.drawActorClass(actor, x2, y);
		this.placeBasicGauges(actor, x2, y + lineHeight * 2);
	};

	Window_Status.prototype.drawExpInfo = function(x, y) {
		const lineHeight = this.lineHeight();
		const expTotal = TextManager.expTotal.format(TextManager.exp);
		const expNext = TextManager.expNext.format(TextManager.level);
		const maxWidth = 270 - 24
		this.changeTextColor(ColorManager.systemColor());
		this.drawText(expTotal, x, y + lineHeight * 0, maxWidth);
		this.drawText(expNext, x, y + lineHeight * 2, maxWidth);
		this.resetTextColor();
		this.drawText(this.expTotalValue(), x, y + lineHeight * 1, maxWidth, "right");
		this.drawText(this.expNextValue(), x, y + lineHeight * 3, maxWidth, "right");
	};

	//-----------------------------------------------------------------------------
	// Window_ProfileTab

	function Window_ProfileTab() {
		this.initialize(...arguments);
	}

	Window_ProfileTab.prototype = Object.create(Window_HorzCommand.prototype);
	Window_ProfileTab.prototype.constructor = Window_ProfileTab;

	const padding = shorterTabWindow ? 6 : 0;
	Window_ProfileTab.prototype.updatePadding = function() {
		this.padding = $gameSystem.windowPadding() - padding;
	};

	Window_ProfileTab.prototype.initialize = function(rect) {
		Window_HorzCommand.prototype.initialize.call(this, rect);
		this._actor = null;
	};

	if (!drawItemFrame) {
		Window_ProfileTab.prototype.drawBackgroundRect = function(rect) {};
	}

	Window_ProfileTab.prototype.makeCommandList = function() {
		const actor = this._actor;
		if (type === 2) {
			this.addCommand(statusText, "status", !!actor);
		}
		this.addCommand(profText, "profile", !!(actor && actor.profile()));
		this.addCommand(classText, "class", !!(actor && actor.currentClass().meta["desc"]));
	};

	Window_ProfileTab.prototype.isOkEnabled = function() {
		return false;
	};

	Window_ProfileTab.prototype.isCancelEnabled = function() {
		return false;
	};

	Window_ProfileTab.prototype.setActor = function(actor) {
		if (this._actor !== actor) {
			this._actor = actor;
			this.refresh();
			this.callUpdateHelp();
		}
	};

	Window_ProfileTab.prototype.actor = function() {
		return this._actor;
	};

	Window_ProfileTab.prototype.updateHelp = function() {
		Window_HorzCommand.prototype.updateHelp.call(this);
		const text = this.profileText();
		this._helpWindow.setText(text);
		if (type === 2) {
			if (this.currentSymbol() === "status") {
				this._helpWindow.hide();
			} else {
				this._helpWindow.show();
			}
		}
	};

	Window_ProfileTab.prototype.profileText = function() {
		const actor = this.actor();
		if (actor) {
			switch (this.currentSymbol()) {
			case "profile":
				return actor.profile();
			case "class":
				return actor.currentClass().meta["desc"] || "";
			}
		}
		return "";
	};

	//-----------------------------------------------------------------------------
	// Window_Profile

	function Window_Profile() {
		this.initialize(...arguments);
	}

	Window_Profile.prototype = Object.create(Window_ProfileTab.prototype);
	Window_Profile.prototype.constructor = Window_Profile;

	Window_Profile.prototype.updatePadding = function() {
		Window_HorzCommand.prototype.updatePadding.call(this);
	};

	Window_Profile.prototype.setText = function() {};

	Window_Profile.prototype.select = function(index) {
		Window_HorzCommand.prototype.select.call(this, index);
		this.refreshProfileText();
	};

	Window_Profile.prototype.refresh = function() {
		Window_ProfileTab.prototype.refresh.call(this);
		if (horzLine) {
			this.drawHorzLine();
		}
		this.refreshProfileText();
	};

	Window_Profile.prototype.drawHorzLine = function() {
		const padding = this.itemPadding();
		const lineHeight = this.lineHeight();
		const itemY = this.itemNameY();
		const profileY = this.profileY();
		const y = Math.floor((itemY + profileY + lineHeight) / 2)-1;
		const width = this.innerWidth;
		const color = ColorManager.normalColor();
		this.contents.paintOpacity = 48;
		this.contents.fillRect(0, y, width, 2, color);
		this.contents.paintOpacity = 255;
	};

	Window_Profile.prototype.itemNameY = function() {
		return this.itemLineRect(0).y;
	};

	Window_Profile.prototype.profileY = function() {
		return this.profileRect().y;
	};

	Window_Profile.prototype.refreshProfileText = function() {
		let text = this.profileText();
		const rect = this.profileRect();
		this.contents.clearRect(rect.x, rect.y, rect.width, rect.height);
		this.drawTextEx(text, rect.x, rect.y, rect.width, rect.height);
	};

	Window_Profile.prototype.profileRect = function() {
		const start = 1;
		const length = 2;
        const rect = this.itemLineRect(start*4).enlarge(this.itemLineRect(start*4+length*4-1));
		rect.y += (this.itemHeight() - Window_Scrollable.prototype.itemHeight.call(this)) * 1;
		return rect;
	};

	//-----------------------------------------------------------------------------
	// Scene_Status

	if (type === 0) {
		const _Scene_Status_create = Scene_Status.prototype.create;
		Scene_Status.prototype.create = function() {
			_Scene_Status_create.call(this);
			this.createProfileTabWindow();
		};

		const _Scene_Status_statusParamsWindowRect = Scene_Status.prototype.statusParamsWindowRect;
		Scene_Status.prototype.statusParamsWindowRect = function() {
			const rect = _Scene_Status_statusParamsWindowRect.call(this);
			rect.y -= this.profileTabHeight();
			return rect;
		};

		const _Scene_Status_statusEquipWindowRect = Scene_Status.prototype.statusEquipWindowRect;
		Scene_Status.prototype.statusEquipWindowRect = function() {
			const rect = _Scene_Status_statusEquipWindowRect.call(this);
			rect.y -= this.profileTabHeight();
			return rect;
		};

		Scene_Status.prototype.createProfileTabWindow = function() {
			const rect = this.profileTabWindowRect();
			this._profileTabWindow = new Window_ProfileTab(rect);
			this._profileTabWindow.setHelpWindow(this._profileWindow);
			this.addWindow(this._profileTabWindow);
		};

		Scene_Status.prototype.profileTabWindowRect = function() {
			const ww = Graphics.boxWidth;
			const wh = this.profileTabHeight();
			const wx = 0;
			const wy = this.profileWindowRect().y - wh;
			return new Rectangle(wx, wy, ww, wh);
		};

		Scene_Status.prototype.profileTabHeight = function() {
			return this.calcWindowHeight(1, true) - padding*2;
		};

		const _Scene_Status_refreshActor = Scene_Status.prototype.refreshActor;
		Scene_Status.prototype.refreshActor = function() {
			_Scene_Status_refreshActor.call(this);
			const actor = this.actor();
			this._profileTabWindow.setActor(actor);
		};
	} else if (type === 1) {

		Scene_Status.prototype.createProfileWindow = function() {
			const rect = this.profileWindowRect();
			this._profileWindow = new Window_Profile(rect);
			this.addWindow(this._profileWindow);
		};

		Scene_Status.prototype.profileHeight = function() {
			return this.calcWindowHeight(3, true);
		};

		const _Scene_Status_refreshActor = Scene_Status.prototype.refreshActor;
		Scene_Status.prototype.refreshActor = function() {
			_Scene_Status_refreshActor.call(this);
			const actor = this.actor();
			this._profileWindow.setActor(actor);
		};
	} else {
		const _Scene_Status_create = Scene_Status.prototype.create;
		Scene_Status.prototype.create = function() {
			_Scene_Status_create.call(this);
			this.createProfileTabWindow();
			this.addWindow(this._profileWindow);
		};

		const _Scene_Status_statusWindowRect = Scene_Status.prototype.statusWindowRect;
		Scene_Status.prototype.statusWindowRect = function() {
			const rect = _Scene_Status_statusWindowRect.call(this);
			rect.height -= this.profileTabHeight()
			return rect;
		};

		const _Scene_Status_statusParamsWindowRect = Scene_Status.prototype.statusParamsWindowRect;
		Scene_Status.prototype.statusParamsWindowRect = function() {
			const rect = _Scene_Status_statusParamsWindowRect.call(this);
			rect.y += this.profileHeight();
			return rect;
		};

		const _Scene_Status_statusEquipWindowRect = Scene_Status.prototype.statusEquipWindowRect;
		Scene_Status.prototype.statusEquipWindowRect = function() {
			const rect = _Scene_Status_statusEquipWindowRect.call(this);
			rect.y += this.profileHeight();
			return rect;
		};

		Scene_Status.prototype.statusParamsHeight = function() {
			return this.calcWindowHeight(numLines, false);
		};

		Scene_Status.prototype.profileHeight = function() {
			return Scene_Status.prototype.statusParamsHeight.call(this);
		};

		Scene_Status.prototype.createProfileTabWindow = function() {
			const rect = this.profileTabWindowRect();
			this._profileTabWindow = new Window_ProfileTab(rect);
			this._profileTabWindow.setHelpWindow(this._profileWindow);
			this.addWindow(this._profileTabWindow);
		};

		Scene_Status.prototype.profileTabWindowRect = function() {
			const ww = Graphics.boxWidth;
			const wh = this.profileTabHeight();
			const wx = 0;
			const wy = this.profileWindowRect().y - wh;
			return new Rectangle(wx, wy, ww, wh);
		};

		Scene_Status.prototype.profileTabHeight = function() {
			return this.calcWindowHeight(1, true) - padding*2;
		};

		const _Scene_Status_refreshActor = Scene_Status.prototype.refreshActor;
		Scene_Status.prototype.refreshActor = function() {
			_Scene_Status_refreshActor.call(this);
			const actor = this.actor();
			this._profileTabWindow.setActor(actor);
		};
	}

}